# -*- coding: utf-8 -*-
"""
Created on Thu Sep 11 11:30:50 2025

@author: mstep
"""
def get_transformations(point_rect, target_rect):
    '''Return transformations to map the coordinate area to the rectangle.'''
    # Translate to center the points at the origin.
    dx1 = -(point_rect[0] + point_rect[2]) / 2
    dy1 = -(point_rect[1] + point_rect[3]) / 2

    # Scale.
    x_scale = (target_rect[2] - target_rect[0]) / (point_rect[2] - point_rect[0])
    y_scale = (target_rect[3] - target_rect[1]) / (point_rect[3] - point_rect[1])
    scale = min(x_scale, y_scale)

    # Translate to move the points to the center of the target rectangle.
    dx2 = (target_rect[0] + target_rect[2]) / 2
    dy2 = (target_rect[1] + target_rect[3]) / 2

    # Return the transformation information.
    return dx1, dy1, scale, dx2, dy2

def get_list_of_lists_bounds(list_of_lists):
    '''Return the coordinate bounds for this list of lists.'''
    xs = [point[0] for point_list in list_of_lists for point in point_list]
    ys = [point[1] for point_list in list_of_lists for point in point_list]
    xmin = min(xs)
    xmax = max(xs)
    ymin = min(ys)
    ymax = max(ys)
    return (xmin, ymin, xmax, ymax)

def transform_list_of_lists(list_of_lists, target_rect):
    '''Transform the list of lists of points to fit the rectangle.'''
    # The list_of_lists parameter is a list of lists of coordinate pairs.
    # For example, it might be a list of segments, squares, or larger polygons.
    # Get the points' coordinate bounds.
    point_rect = get_list_of_lists_bounds(list_of_lists)

    # Get the transformations.
    dx1, dy1, scale, dx2, dy2 = get_transformations(point_rect, target_rect)

    # Apply the transformations to the points.
    # Using a list comprehension:
    new_list_of_lists = [[
        (
            (point[0] + dx1) * scale + dx2,
            (point[1] + dy1) * scale + dy2,
        ) for point in point_list] for point_list in list_of_lists]
    # Using loops:
    # new_list_of_lists = []
    # for point_list in list_of_lists:
    #     new_point_list = []
    #     new_list_of_lists.append(new_point_list)
    #     for point in point_list:
    #         x = (point[0] + dx1) * scale + dx2
    #         y = (point[1] + dy1) * scale + dy2
    #         new_point_list.append((x, y))

    return new_list_of_lists

#%%
import math

def get_binary_tree(depth, delta, length, scale, x, y, angle=-math.pi/2,
                    segments=None):
    '''Return a list of segments that draw a binary tree.'''
    if segments is None:
        segments = []

    # Draw our segment.
    x1 = x + length * math.cos(angle)
    y1 = y + length * math.sin(angle)
    segments.append(((x, y), (x1, y1)))

    # If depth > 0, recurse.
    if depth > 0:
        get_binary_tree(depth - 1, delta, length * scale, scale, x1, y1,
                        angle - delta, segments)
        get_binary_tree(depth - 1, delta, length * scale, scale, x1, y1,
                        angle + delta, segments)

    # Return the segments list.
    return segments

#%%
import tkinter as tk
import math

class BinaryTreeFittedApp:
    def __init__(self):
        self.window = tk.Tk()
        self.window.title('binary_tree_fitted')
        self.window.protocol('WM_DELETE_WINDOW', self.kill_callback)

        # Make the control widgets.
        top_frame = tk.Frame(self.window, width=50)
        top_frame.pack(side=tk.TOP, padx=(5,0), pady=(5,0), anchor='nw',
                       fill=tk.X)

        wid = 5
        label = tk.Label(top_frame, text='Depth:')
        label.pack(side=tk.LEFT, padx=(10,0))
        self.depth_entry = tk.Entry(top_frame, width=wid, justify='right')
        self.depth_entry.pack(side=tk.LEFT)
        self.depth_entry.insert(0, '10')

        label = tk.Label(top_frame, text='Delta:')
        label.pack(side=tk.LEFT, padx=(10,0))
        self.delta_entry = tk.Entry(top_frame, width=wid, justify='right')
        self.delta_entry.pack(side=tk.LEFT)
        self.delta_entry.insert(0, '45')

        label = tk.Label(top_frame, text='Scale:')
        label.pack(side=tk.LEFT, padx=(10,0))
        self.scale_entry = tk.Entry(top_frame, width=wid, justify='right')
        self.scale_entry.pack(side=tk.LEFT)
        self.scale_entry.insert(0, '0.75')

        button = tk.Button(top_frame, text='Draw', width=7, command=self.draw)
        button.pack(side=tk.LEFT, padx=(20,0))

        # Make the canvas.
        self.canvas = tk.Canvas(self.window, bg='white',
            borderwidth=2, relief=tk.SUNKEN, width=600, height=400)
        self.canvas.pack(padx=5, pady=5,
            side=tk.TOP, fill=tk.BOTH, expand=True)

        # Make Enter trigger a draw.
        self.window.bind('<Return>', lambda event: self.draw())

        # Draw the initial tree.
        self.draw()

        # Display the window.
        self.depth_entry.focus_force()
        self.window.mainloop()

    def kill_callback(self):
        # Destroy the tkinter window.
        self.window.destroy()

    def draw(self):
        '''Draw the tree.'''
        # Clear the canvas.
        self.canvas.delete(tk.ALL)

        # Get the tree's parameters.
        depth = int(self.depth_entry.get())
        delta = math.radians(float(self.delta_entry.get()))
        length = 1
        scale = float(self.scale_entry.get())
        x = 0
        y = 0

        # Generate the tree segments.
        segments = get_binary_tree(depth, delta, length, scale, x, y)

        # Get the drawing area coordinates and set the target rectangle.
        tk.Tk.update(self.canvas)
        canvas_wid = self.canvas.winfo_width()
        canvas_hgt = self.canvas.winfo_height()
        margin = 10
        rect = (margin, margin, canvas_wid - margin, canvas_hgt - margin)

        # Transform the points so they fit the target area.
        segments = transform_list_of_lists(segments, rect)

        # Draw the segments.
        for segment in segments:
            self.canvas.create_line(segment[0], segment[1], fill='blue')

#%%
BinaryTreeFittedApp()
