# -*- coding: utf-8 -*-
"""
Created on Thu Sep 11 11:30:50 2025

@author: mstep
"""
import pygame

def get_transformations(point_lists, target_rect):
    '''Return transformations to map the Vector2 points to the rectangle.'''
    # Get the points' bounds.
    p1 = point_lists[0][0]
    xmin = xmax = p1.x
    ymin = ymax = p1.y
    for rect in point_lists:
        for point in rect:
            if xmin > point.x: xmin = point.x
            if xmax < point.x: xmax = point.x
            if ymin > point.y: ymin = point.y
            if ymax < point.y: ymax = point.y

    # Translate to center the points at the origin.
    dx1 = -(xmin + xmax) / 2
    dy1 = -(ymin + ymax) / 2
    translate1 = pygame.Vector2(dx1, dy1)

    # Scale.
    x_scale = (target_rect[2] - target_rect[0]) / (xmax - xmin)
    y_scale = (target_rect[3] - target_rect[1]) / (ymax - ymin)
    scale = min(x_scale, y_scale)

    # Translate to move the points to the center of the target rectangle.
    dx2 = (target_rect[0] + target_rect[2]) / 2
    dy2 = (target_rect[1] + target_rect[3]) / 2
    translate2 = pygame.Vector2(dx2, dy2)

    # Return the transformation information.
    return translate1, scale, translate2

def transform_point_lists(point_lists, rect):
    '''Transform the list of point lists to fit the rect.'''
    # Get the transformations.
    translate1, scale, translate2 = get_transformations(point_lists, rect)

    # Apply the transformations.
    transformed_lists = [
        [(point + translate1) * scale + translate2
             for point in point_list]
                for point_list in point_lists]
    return transformed_lists

#%%
def get_fibonacci_squares(num_squares):
    '''Return a list of Vector2 points forming Fibonacci squares.'''
    # Initialize the first 2 squares.
    squares = [
        [(0, 0), (0, -1), (1, -1), (1, 0)],
        [(1, -1), (2, -1), (2, 0), (1, 0)],
    ]
    squares = [[pygame.Vector2(point) for point in rect]
                  for rect in squares]

    # Generate other squares.
    for i in range(2, num_squares):
        # Make square number i (with numbers starting at 0).
        # It shares an edge with squares[i - 1] and squares[i - 2].
        # p0 is at the upper left corner of the previous square.
        p0 = squares[i - 1][2]

        # Get the direction of the first side.
        v01 = squares[i - 1][2] - squares[i - 1][1]
        v01.scale_to_length(fibonacci(i + 1))

        # Get the second point.
        p1 = p0 + v01

        # Rotate v01 by 90 degrees to get the next side direction.
        v12 = v01.rotate(90)

        # Get the third and fourth points.
        p2 = p1 + v12
        p3 = p2 - v01

        # Save the square.
        squares.append((p0, p1, p2, p3))

    # Flip the Y coordinates.
    squares = [[pygame.Vector2(point.x, -point.y) for point in rect]
                  for rect in squares]

    return squares

def fibonacci(n):
    '''Return the nth fibonacci number.'''
    if n <= 1:
        return n

    # Calculate larger values.
    fib_i_minus_2 = 0
    fib_i_minus_1 = 1
    for i in range(2, n+1):
        new_value = fib_i_minus_1 + fib_i_minus_2
        fib_i_minus_2 = fib_i_minus_1
        fib_i_minus_1 = new_value
    return new_value

def polygon_bounds(polygon):
    '''Return the polygon's minimum and maximum X and Y coordinates.'''
    xmin = xmax = polygon[0][0]
    ymin = ymax = polygon[0][1]
    for point in polygon:
        if point[0] < xmin: xmin = point[0]
        if point[0] > xmax: xmax = point[0]
        if point[1] < ymin: ymin = point[1]
        if point[1] > ymax: ymax = point[1]

    return xmin, ymin, xmax, ymax

#%%
import tkinter as tk
from tkinter import ttk
import math

class FibonacciSpiralApp:
    def __init__(self):
        self.window = tk.Tk()
        self.window.title('fibonacci_spiral')
        self.window.protocol('WM_DELETE_WINDOW', self.kill_callback)

        # Make the control widgets.
        top_frame = tk.Frame(self.window, width=50)
        top_frame.pack(side=tk.TOP, padx=(5,0), pady=(5,0), anchor='nw',
                       fill=tk.X)

        label = tk.Label(top_frame, text='# squares:')
        label.pack(side=tk.LEFT)

        self.num_squares_var = tk.IntVar(value=8)
        spinbox = tk.Spinbox(top_frame, from_=2, to=40, increment=1, width=4,
                             textvariable=self.num_squares_var,
                             justify=tk.RIGHT, command=self.draw)
        spinbox.pack(side=tk.LEFT)

        label = tk.Label(top_frame, text='Text:')
        label.pack(side=tk.LEFT, padx=(20,0))

        values = ['Sizes', 'Indexes', 'None']
        self.text_var = tk.StringVar(value=values[0])
        combo = ttk.Combobox(top_frame, textvariable=self.text_var,
                            values=values)
        combo.set(values[0])
        combo.pack(side=tk.LEFT)
        combo.bind('<<ComboboxSelected>>', lambda event:self.draw())

        self.draw_squares_var = tk.BooleanVar(value=True)
        checkbutton = tk.Checkbutton(top_frame, text='Draw Squares',
                                     variable=self.draw_squares_var,
                                     command=self.draw)
        checkbutton.pack(side=tk.LEFT, padx=(20,0))

        # Make the canvas.
        self.canvas = tk.Canvas(self.window, bg='white',
            borderwidth=2, relief=tk.SUNKEN, width=600, height=400)
        self.canvas.pack(padx=5, pady=5,
            side=tk.TOP, fill=tk.BOTH, expand=True)

        # Draw the initial Pythagoras tree.
        self.draw()

        # Display the window.
        spinbox.focus_force()
        self.window.mainloop()

    def kill_callback(self):
        # Destroy the tkinter window.
        self.window.destroy()

    def draw(self):
        '''Draw the Fibonacci squares.'''
        # Get the number of squares.
        num_squares = self.num_squares_var.get()

        # Get the drawing area coordinates.
        tk.Tk.update(self.canvas)
        canvas_wid = self.canvas.winfo_width()
        canvas_hgt = self.canvas.winfo_height()

        # Get the squares.
        squares = get_fibonacci_squares(num_squares)

        # Transform the squares to fit the canvas.
        margin = 10
        rect = (margin, margin, canvas_wid - margin, canvas_hgt - margin)
        squares = transform_point_lists(squares, rect)

        # Convert the Vector2 objects to (x, y) coordinate pairs.
        squares = [[(point.x, point.y) for point in square]
                          for square in squares]

        # See what text we should display.
        text_to_draw = self.text_var.get()

        # Draw the squares.
        self.canvas.delete(tk.ALL)
        outline = 'blue'
        for i, square in enumerate(squares):
            # Get the text's angle.
            angle = 90 * ((i + 2) % 4)

            # See if we should draw the squares.
            if self.draw_squares_var.get():
                fill=''
                self.canvas.create_polygon(square, fill=fill, outline=outline)

                # Get the square's centroid.
                cx = (square[0][0] + square[1][0] +
                      square[2][0] + square[3][0]) / 4
                cy = (square[0][1] + square[1][1] +
                      square[2][1] + square[3][1]) / 4

                # Set the text.
                if text_to_draw == 'Sizes':
                    # Display each square's size, which is its Fibonacci number.
                    p = fibonacci(i + 1)
                    text = f'{p}'
                elif text_to_draw == 'Indexes':
                    # Display the index.
                    text = f'{i}'
                else:
                    text = ''

                # Get the square's height.
                base = math.dist(square[0], square[1])
                hgt = base / math.sqrt(3)
                num_digits = len(text)
                font_size = int(hgt / (num_digits + 0.5))

                # Draw the text.
                self.canvas.create_text(cx, cy, text=text, fill='black',
                                        font=('Arial', font_size, 'bold'),
                                        angle=angle)

            # Draw the spiral's arc.
            # Get a square twice as wide and tall as this one.
            v01 = (square[1][0] - square[0][0],
                   square[1][1] - square[0][1])
            v12 = (square[2][0] - square[1][0],
                   square[2][1] - square[1][1])
            p0 = (square[0][0] - v01[0], square[0][1]- v01[1])
            p1 = (p0[0] + 2 * v01[0], p0[1] + 2 * v01[1])
            p2 = (p1[0] + 2 * v12[0], p1[1] + 2 * v12[1])
            p3 = (p2[0] - 2 * v01[0], p2[1] - 2 * v01[1])
            xmin, ymin, xmax, ymax = polygon_bounds((p0, p1, p2, p3))

            # Draw the arc.
            self.canvas.create_arc(xmin, ymin, xmax, ymax,
                                   start=angle, extent=90, style=tk.ARC,
                                   fill='', outline='purple', width=2)

        # Draw the origin.
        origin = squares[0][0]
        radius = (squares[0][2][0] - squares[0][0][0]) / 10
        self.canvas.create_oval(
            origin[0] - radius, origin[1] - radius,
            origin[0] + radius, origin[1] + radius,
            fill='black')


#%%
FibonacciSpiralApp()
