# -*- coding: utf-8 -*-
"""
Created on Thu Sep 11 11:30:50 2025

@author: mstep
"""
#%%
import tkinter as tk
from tkinter import ttk
import math

class LifeApp:
    def __init__(self):
        self.window = tk.Tk()
        self.window.title('life')
        self.window.protocol('WM_DELETE_WINDOW', self.kill_callback)

        # Load preset paterns.
        self.load_patterns()

        # Make the control widgets.
        top_frame = tk.Frame(self.window, width=50)
        top_frame.pack(side=tk.TOP, padx=(5,0), anchor='nw', fill=tk.X)

        self.pattern_combo = \
            ttk.Combobox(top_frame, width=22, state='readonly',
                         values=list(self.patterns.keys()))
        self.pattern_combo.bind('<<ComboboxSelected>>', self.load_pattern)
        self.pattern_combo.pack(side=tk.LEFT, padx=3, anchor=tk.S)
        self.pattern_combo.current(0)

        label = tk.Label(top_frame, text='Speed (FPS):')
        label.pack(side=tk.LEFT, padx=(10,0), anchor=tk.S)

        self.fps_scale = \
            tk.Scale(top_frame, from_=1, to=100, orient=tk.HORIZONTAL,
                     resolution=1)
        self.fps_scale.pack(side=tk.LEFT, padx=(0,10),
                              expand=True, fill=tk.X)

        self.start_button = tk.Button(top_frame, text='Start', width=10,
                                      command=self.start_stop)
        self.start_button.pack(side=tk.RIGHT, padx=7, anchor=tk.S)

        self.window.bind('<Return>', lambda event: self.draw())

        self.status_label = tk.Label(self.window, text='')
        self.status_label.pack(side=tk.BOTTOM, anchor=tk.SW, padx=5, pady=(0,5))

        # Make the canvas.
        self.canvas = tk.Canvas(self.window, bg='white', cursor='crosshair',
            borderwidth=2, relief=tk.SUNKEN, width=500, height=500)
        self.canvas.pack(padx=5, pady=(5,0),
            side=tk.TOP, fill=tk.BOTH, expand=True)
        self.canvas.bind('<ButtonPress-1>', self.mouse_down) 

        # Make the world.
        self.make_world()
        self.running = False

        # Display the window.
        self.window.focus_force()
        self.window.mainloop()

    def kill_callback(self):
        # Destroy the tkinter window.
        self.window.destroy()

    def load_patterns(self):
        '''Load known patterns.'''
        self.patterns = {
            'Clear': [],
            'Block': ['11', '11'],
            'Beehive': ['0110', '1001', '0110'],
            'Loaf': ['0110', '1001', '0101', '0010'],
            'Boat': ['110', '101', '010'],
            'Tub': ['010', '101', '010'],
            'Blinker': ['010', '010', '010'],
            'Stoplight': ['010', '010', '010', '000', '000', '111', '000',
                          '000', '010', '010', '010'],
            'Toad': ['0111', '1110'],
            'Beacon': ['1100', '1100', '0011', '0011'],
            'Pulsar': ['0011100011100', '0000000000000', '1000010100001',
                       '1000010100001', '1000010100001', '0011100011100',
                       '0000000000000', '0011100011100', '1000010100001',
                       '1000010100001', '1000010100001', '0000000000000',
                       '0011100011100'],
            'Pentadecathlon': ['01110', '10001', '10001', '01110', '00000',
                               '00000', '00000', '00000', '01110', '10001',
                               '10001', '01110'],
            'Glider': ['010', '001', '111'],
            'Lightweight Spaceship': ['10010', '00001', '10001', '01111'],
            'R-Pentomino': ['011', '110', '010'],
            'Diehard': ['00000010', '11000000', '01000111'],
            'Acorn': ['0100000', '0001000', '1100111'],
            'Gosper Glider Gun': [
                '000000000000000000000000100000000000',
                '000000000000000000000010100000000000',
                '000000000000110000001100000000000011',
                '000000000001000100001100000000000011',
                '110000000010000010001100000000000000',
                '110000000010001011000010100000000000',
                '000000000010000010000000100000000000',
                '000000000001000100000000000000000000',
                '000000000000110000000000000000000000'],
            'Pattern 1': ['00000010', '00001011', '00001010', '00001000',
                         '00100000', '10100000'],
            'Pattern 2': ['11101', '10000', '00011', '01101', '10101'],
            'Pattern 3': ['111111110111110001110000001111111011111'],
        }

    def make_world(self):
        '''Make the world.'''
        # Get the canvas's dimensions.
        tk.Tk.update(self.canvas)
        self.wid = self.canvas.winfo_width()
        self.hgt = self.canvas.winfo_height()
        self.dx = 10
        self.dy = 10
        self.num_rows = math.ceil(self.wid / self.dx)
        self.num_cols = math.ceil(self.hgt / self.dy)

        # Make the world lists.
        self.world = [[False for c in range(self.num_cols)]
                          for r in range(self.num_rows)]
        self.temp  = [[False for c in range(self.num_cols)]
                          for r in range(self.num_rows)]

        # Make the squares.        
        self.squares = []
        y = 0
        for r in range(self.num_rows):
            row = []
            self.squares.append(row)
            x = 0
            for c in range(self.num_cols):
                row.append(self.canvas.create_rectangle(
                    x, y, x + self.dx, y + self.dy,
                    fill='white', outline='black'))
                x += self.dx
            y += self.dy

    def clear_world(self):
        '''Clear the whole world.'''
        for r in range(len(self.world)):
            for c in range(len(self.world[r])):
                if self.world[r][c]:
                    self.set_square(r, c, False)

    def load_pattern(self, event):
        '''Stop and load the selected pattern.'''
        self.start_button.config(text='Start')
        self.running = False
        self.num_steps = 0
        self.status_label['text'] = ''

        # Clear the world.
        self.clear_world()

        # Get the pattern.
        pattern = self.patterns[self.pattern_combo.get()]

        # Load the pattern.
        num_rows = len(pattern)
        if num_rows == 0:   # If the pattern is Clear.
            return
        num_cols = len(pattern[0])
        min_r = (len(self.world) - num_rows) // 2
        min_c = (len(self.world[0]) - num_cols) // 2
        for r in range(len(pattern)):
            for c in range(len(pattern[r])):
                if pattern[r][c] == '1':
                    self.set_square(r + min_r, c + min_c, True)

    def mouse_down(self, event):
        '''Toggle the square the user clicked.'''
        self.start_button.config(text='Start')
        self.running = False
        c = event.x // self.dx
        r = event.y // self.dy
        self.set_square(r, c, not self.world[r][c])
        self.num_steps = 0
        self.status_label['text'] = ''

    def start_stop(self):
        '''Start or stop.'''
        # See if the button says Start or Stop.
        if self.start_button.cget('text') == 'Start':
            # Start.
            self.start_button.config(text='Stop')
            self.running = True
            self.step()
        else:
            # Stop.
            self.start_button.config(text='Start')
            self.running = False

    def step(self):
        '''Make a game step.'''
        # If we're no longer running, stop.
        if not self.running:
            return

        # Update the living squares in the temp lists.
        for r in range(self.num_rows):
            for c in range(self.num_cols):
                self.update_square(r, c)

        # Copy new values into the world lists and update square colors.
        for r in range(self.num_rows):
            for c in range(self.num_cols):
                self.set_square(r, c, self.temp[r][c])

        # Update the turn label.
        self.num_steps += 1
        self.status_label['text'] = f'Step {self.num_steps}'

        # Schedule the next step.
        fps = self.fps_scale.get()
        delay = 1000 // fps
        self.window.after(delay, self.step)

    def update_square(self, r, c):
        '''Update this square.'''
        # Count the neighbors.
        num_neighbors = self.count_neighbors(r, c)

        # See if this square is currently alive. Save new values in the
        # temp lists and copy them into the world lists when we're done.
        if self.world[r][c]:
            # Alive.
            if num_neighbors < 2:
                # Underpopulation.
                self.temp[r][c] = False
            elif num_neighbors > 3:
                # Overpopulation.
                self.temp[r][c] = False
            else:
                # Status quo.
                self.temp[r][c] = True
        else:
            # Vacant.
            if num_neighbors == 3:
                # Reproduction.
                self.temp[r][c] = True
            else:
                # Status quo.
                self.temp[r][c] = False

    def count_neighbors(self, r, c):
        '''Return the number of neighbors this square has.'''
        num_neighbors = 0
        for nbr_r in [r - 1, r, r + 1]:
            if nbr_r < 0 or nbr_r >= self.num_rows:
                continue
            for nbr_c in [c - 1, c, c + 1]:
                if nbr_r == r and nbr_c == c:
                    continue
                if nbr_c < 0 or nbr_c >= self.num_cols:
                    continue
                if self.world[nbr_r][nbr_c]:
                    num_neighbors += 1
        return num_neighbors

    def set_square(self, r, c, is_alive):
        '''Set the world entry and the cell's color.'''
        # See if the value needs to be changed.
        if not self.world[r][c] == is_alive:
            # Update the cell and its square.
            self.world[r][c] = is_alive
            color = 'black' if is_alive else 'white'
            self.canvas.itemconfigure(self.squares[r][c], fill=color)

#%%
LifeApp()
