# -*- coding: utf-8 -*-
"""
Created on Thu Sep 11 11:30:50 2025

@author: mstep
"""
import pygame

def get_transformations(point_lists, target_rect):
    '''Return transformations to map the Vector2 points to the rectangle.'''
    # Get the points' bounds.
    p1 = point_lists[0][0]
    xmin = xmax = p1.x
    ymin = ymax = p1.y
    for rect in point_lists:
        for point in rect:
            if xmin > point.x: xmin = point.x
            if xmax < point.x: xmax = point.x
            if ymin > point.y: ymin = point.y
            if ymax < point.y: ymax = point.y

    # Translate to center the points at the origin.
    dx1 = -(xmin + xmax) / 2
    dy1 = -(ymin + ymax) / 2
    translate1 = pygame.Vector2(dx1, dy1)

    # Scale.
    x_scale = (target_rect[2] - target_rect[0]) / (xmax - xmin)
    y_scale = (target_rect[3] - target_rect[1]) / (ymax - ymin)
    scale = min(x_scale, y_scale)

    # Translate to move the points to the center of the target_rectangle.
    dx2 = (target_rect[0] + target_rect[2]) / 2
    dy2 = (target_rect[1] + target_rect[3]) / 2
    translate2 = pygame.Vector2(dx2, dy2)

    # Return the transformation information.
    return translate1, scale, translate2

def transform_point_lists(point_lists, rect):
    '''Transform the list of point lists to fit the rect.'''
    # Get the transformations.
    translate1, scale, translate2 = get_transformations(point_lists, rect)

    # Apply the transformations.
    transformed_lists = [
        [(point + translate1) * scale + translate2
             for point in point_list]
                for point_list in point_lists]
    return transformed_lists

#%%
def get_padovan_triangles(num_triangles):
    '''Return a list of Vector2 points forming Padovan triangles.'''
    triangles = []

    # Find the first 7 points.
    p0 = pygame.Vector2(0, 0)
    p1 = pygame.Vector2(0.5, -0.5 * math.sqrt(3))
    p2 = pygame.Vector2(1, 0)
    p3 = pygame.Vector2(0.5, 0.5 * math.sqrt(3))
    p4 = pygame.Vector2(-0.5, 0.5 * math.sqrt(3))
    p5 = pygame.Vector2(-1.5, -0.5 * math.sqrt(3))
    p6 = pygame.Vector2(-0.5, -3 * 0.5 * math.sqrt(3))

    # Make the first 5 triangles.
    triangles.append((p0, p1, p2))
    triangles.append((p0, p2, p3))
    triangles.append((p0, p3, p4))
    triangles.append((p1, p4, p5))
    triangles.append((p1, p5, p6))

    # Generate other triangles.
    for i in range(5, num_triangles):
        # Make triangle number i (with numbers starting at 0).
        # It shares an edge with triangles[i - 1] and triangles[i - 5].
        triangles.append(get_triangle(triangles[i-5][2], triangles[i-1][2]))

    return triangles

def get_triangle(p0, p1):
    '''Return an equilateral triangle with this upper right side.'''
    # Find the triangles' vertices as Vector2 objects.
    v01 = p1 - p0
    v02 = v01.rotate(60)
    p2 = p0 + v02

    # Return the points as (x, y) coordinate pairs.
    return (p0, p1, p2)

def padovan(i):
    '''Return the ith Padovan number.'''
    if i <= 2:
        return 1
    return padovan(i - 2) + padovan(i - 3)

#%%
import tkinter as tk
from tkinter import ttk
import math

class PadovanTrianglesApp:
    def __init__(self):
        self.window = tk.Tk()
        self.window.title('padovan_triangles')
        self.window.protocol('WM_DELETE_WINDOW', self.kill_callback)

        # Make the control widgets.
        top_frame = tk.Frame(self.window, width=50)
        top_frame.pack(side=tk.TOP, padx=(5,0), pady=(5,0), anchor='nw',
                       fill=tk.X)

        label = tk.Label(top_frame, text='# Triangles:')
        label.pack(side=tk.LEFT)

        self.num_triangles_var = tk.IntVar(value=7)
        spinbox = tk.Spinbox(top_frame, from_=5, to=40, increment=1, width=4,
                             textvariable=self.num_triangles_var,
                             justify=tk.RIGHT, command=self.draw)
        spinbox.pack(side=tk.LEFT)

        label = tk.Label(top_frame, text='Text:')
        label.pack(side=tk.LEFT, padx=(20,0))

        values = ['Sizes', 'Indexes', 'None']
        self.text_var = tk.StringVar(value=values[0])
        combo = ttk.Combobox(top_frame, textvariable=self.text_var,
                             values=values)
        combo.set(values[0])
        combo.pack(side=tk.LEFT)
        combo.bind('<<ComboboxSelected>>', lambda event:self.draw())

        # Make the canvas.
        self.canvas = tk.Canvas(self.window, bg='white',
            borderwidth=2, relief=tk.SUNKEN, width=600, height=400)
        self.canvas.pack(padx=5, pady=5,
            side=tk.TOP, fill=tk.BOTH, expand=True)

        # Draw the initial Pythagoras tree.
        self.draw()

        # Display the window.
        spinbox.focus_force()
        self.window.mainloop()

    def kill_callback(self):
        # Destroy the tkinter window.
        self.window.destroy()

    def draw(self):
        '''Draw the Padovan triangles.'''
        # Get the number of triangles.
        num_triangles = self.num_triangles_var.get()

        # Get the drawing area coordinates.
        tk.Tk.update(self.canvas)
        canvas_wid = self.canvas.winfo_width()
        canvas_hgt = self.canvas.winfo_height()

        # Get the triangles.
        triangles = get_padovan_triangles(num_triangles)

        # Transform the triangles to fit the canvas.
        margin = 10
        rect = (margin, margin, canvas_wid - margin, canvas_hgt - margin)
        triangles = transform_point_lists(triangles, rect)

        # Convert the Vector2 objects to (x, y) coordinate pairs.
        triangles = [[(point.x, point.y) for point in triangle]
                          for triangle in triangles]

        # See what text we should display.
        text_to_draw = self.text_var.get()

        # Draw the triangles.
        self.canvas.delete(tk.ALL)
        outline = 'blue'
        for i, triangle in enumerate(triangles):
            fill = 'white' if i % 2 == 0 else 'light blue'
            self.canvas.create_polygon(triangle, fill=fill, outline=outline)

            # Get the triangle's centroid.
            cx = (triangle[0][0] + triangle[1][0] + triangle[2][0]) / 3
            cy = (triangle[0][1] + triangle[1][1] + triangle[2][1]) / 3

            # Set the text.
            p = padovan(i)
            if text_to_draw == 'Sizes':
                # Display each square's size, which is its Fibonacci number.
                text = f'{p}'
            elif text_to_draw == 'Indexes':
                # Display the index.
                text = f'{i}'
            else:
                text = ''

            # Get the triangle's height.
            base = math.dist(triangle[0], triangle[1])
            hgt = base / math.sqrt(3)
            num_digits = len(f'{p}')
            font_size = int(hgt / (num_digits + 0.5))

            # Calculate the text's angle.
            angle = 120 - i * 60

            # Draw the text.
            self.canvas.create_text(cx, cy, text=text, fill='blue',
                            angle=angle, font=('Arial', font_size, 'bold'))

        # Draw the origin.
        origin = triangles[0][0]
        radius = (triangles[0][2][0] - triangles[0][0][0]) / 10
        self.canvas.create_oval(
            origin[0] - radius, origin[1] - radius,
            origin[0] + radius, origin[1] + radius,
            fill='black')

#%%
PadovanTrianglesApp()
