# -*- coding: utf-8 -*-
"""
Created on Thu Oct 11 11:30:50 2025

@author: mstep
"""
import tkinter.font as tkFont

def vectors_to_coords(vectors):
    '''Convert a list of pygameVector2 objects into a list of X, Y pairs.'''
    return [(vector.x, vector.y) for vector in vectors]

def draw_vector(canvas, start_point, vector, text=None, fill='black'):
    '''Draw a pygame.Vector2.'''
    width = 1
    arrowhead_size = 5

    # Draw the start point.
    radius = 3 * width
    canvas.create_oval(
        start_point.x - radius, start_point.y - radius,
        start_point.x + radius, start_point.y + radius,
        fill=fill, outline=fill)

    # Get arrowhead parameters.
    tip = start_point + vector
    dy = vector.normalize() * arrowhead_size
    dx = dy.rotate(90)

    # Draw the vector.
    canvas.create_line(start_point.x, start_point.y, tip.x, tip.y,
                       fill=fill, width=width)

    # Draw the arrowhead.
    arrowhead_points = [
        tip + dx - 2 * dy,
        tip,
        tip - dx - 2 * dy,
    ]
    arrowhead_points = vectors_to_coords(arrowhead_points)
    canvas.create_line(arrowhead_points, fill=fill, width=width)

    # Label the vector.
    if text is not None:
        # Get the text's size.
        font = tkFont.Font(family='courier', size=14, weight='bold')
        rx = font.measure(text) / 2 + 3
        ry = font.metrics('linespace') / 2
        center = (start_point + tip) / 2
        canvas.create_rectangle(
            center.x - rx, center.y - ry,
            center.x + rx, center.y + ry,
            outline='', fill='white')
        canvas.create_text(center.x, center.y, text=text, anchor='c',
                           fill=fill, font=('Courier', 14))

#%%
import tkinter as tk
import pygame

class Vector2App:
    def __init__(self):
        self.window = tk.Tk()
        self.window.title('pygame_vector2')
        self.window.protocol('WM_DELETE_WINDOW', self.kill_callback)

        # Make the canvas.
        self.canvas = tk.Canvas(self.window, bg='white',
            borderwidth=2, relief=tk.SUNKEN, width=600, height=400)
        self.canvas.pack(padx=5, pady=5,
            side=tk.TOP, fill=tk.BOTH, expand=True)

        # Draw.
        self.draw()

        # Display the window.
        self.window.focus_force()
        self.window.mainloop()

    def kill_callback(self):
        # Destroy the tkinter window.
        self.window.destroy()

    def draw(self):
        '''Draw example vectors.'''
        # Draw a + b = c.
        a = pygame.Vector2(50, 100)
        b = pygame.Vector2(100, -30)
        c = a + b

        x = 20
        origin = pygame.Vector2(x, 20)
        draw_vector(self.canvas, origin, a, 'a', 'green')

        point = origin + a
        draw_vector(self.canvas, point, b, 'b', 'blue')

        draw_vector(self.canvas, origin, c, 'c', 'red')

        # Draw a and 2 * a.
        x = 250
        origin = pygame.Vector2(x, 50)
        draw_vector(self.canvas, origin, a, 'a', 'green')

        x += 50
        origin = pygame.Vector2(x, 50)
        draw_vector(self.canvas, origin, 2 * a, '2 * a', 'blue')

        # Draw a and -a.
        x += 200
        origin = pygame.Vector2(x, 150)
        draw_vector(self.canvas, origin, a, 'a', 'green')
        draw_vector(self.canvas, origin, -a, '-a', 'blue')

        # Draw f = a + b + d + e.
        x = 20
        origin = pygame.Vector2(x, 200)
        draw_vector(self.canvas, origin, a, 'a', 'green')
        point = origin + a
        draw_vector(self.canvas, point, b, 'b', 'blue')
        d = pygame.Vector2(100, 100)
        point += b
        draw_vector(self.canvas, point, d, 'd', 'purple')
        e = pygame.Vector2(20, -100)
        point += d
        draw_vector(self.canvas, point, e, 'e', 'orange')

        f = a + b + d + e
        draw_vector(self.canvas, origin, f, 'f', 'red')

#%%
Vector2App()
