# -*- coding: utf-8 -*-
"""
Created on Thu Nov  7 15:31:13 2024

@author: mstep
"""
def rgb_to_tk_color(rgb):
    '''Convert an RGB tuple into a color that tkinter understands.'''
    return f'#{rgb[0]:02X}{rgb[1]:02X}{rgb[2]:02X}'

def draw_text(canvas, padx, pady, text, fill, position):
    '''Draw text with the desired position.'''
    # Get the canvas's dimensions.
    tk.Tk.update(canvas)
    canvas_wid = canvas.winfo_width()
    canvas_hgt = canvas.winfo_height()

    # Draw the initial text and get its size.
    text_id = canvas.create_text(0, 0, text=text, fill=fill, anchor='nw')
    bbox = canvas.bbox(text_id)
    text_wid = bbox[2] - bbox[0]
    text_hgt = bbox[3] - bbox[1]

    # Find the text's position.
    position = position.lower()
    if 'n' in position:
        y = pady
    elif 's' in position:
        y = canvas_hgt - text_hgt - pady
    else:
        y = (canvas_hgt - text_hgt) / 2

    if 'w' in position:
        x = padx
    elif 'e' in position:
        x = canvas_wid - text_wid - padx
    else:
        x = (canvas_wid - text_wid) / 2

    # Position the text.
    canvas.move(text_id, x, y)

#%%
import tkinter as tk
from tkinter import messagebox
from PIL import Image

class MoveBorderlessWindowApp:
    def __init__(self):
        self.window = tk.Tk()
        self.window.title('skinned_form_part4')
        self.window.protocol('WM_DELETE_WINDOW', self.kill_callback)
        self.window.geometry('400x300')

        # Remove the window's borders.
        self.window.overrideredirect(True)
        self.window.configure(borderwidth=0, highlightthickness=0)

        # Make the window topmost.
        # (It's not in the task list so otherwise it's hard to find.)
        self.window.attributes("-topmost", True)

        # Make cyan pieces tansparent.
        self.window.wm_attributes('-transparentcolor', 'cyan')
        self.window.configure(bg='cyan') 

        # This is used when the user clicks the kill area.
        self.ignore_drag = False

        # Resizing rectangles.
        size = 10

        # Top.
        top_frame = tk.Frame(self.window, bg='cyan', highlightthickness=0)
        top_frame.pack(side=tk.TOP, fill=tk.X)
        self.nw_canvas = tk.Canvas(top_frame, bg='yellow',
                                   cursor='top_left_corner',
                                   width=3*size, height=3*size,
                                   highlightthickness=0)
        self.nw_canvas.pack(side=tk.LEFT)

        self.ne_canvas = tk.Canvas(top_frame, bg='yellow',
                                   cursor='top_right_corner',
                                   width=3*size, height=3*size,
                                   highlightthickness=0)
        self.ne_canvas.pack(side=tk.RIGHT)

        self.n_canvas = tk.Canvas(top_frame, bg='orange',
                                  cursor='top_side',
                                  height=size, highlightthickness=0)
        self.n_canvas.pack(side=tk.TOP, fill=tk.X)

        self.drag_canvas = tk.Canvas(top_frame, bg='light blue',
                                     cursor='fleur',
                                     height=2*size, highlightthickness=0)
        self.drag_canvas.pack(side=tk.TOP, fill=tk.X)

        # Bottom.
        bottom_frame = tk.Frame(self.window, bg='cyan', highlightthickness=0)
        bottom_frame.pack(side=tk.BOTTOM, fill=tk.X)

        self.sw_canvas = tk.Canvas(bottom_frame, bg='yellow',
                                   cursor='bottom_left_corner',
                                   width=size, height=size)
        self.sw_canvas.pack(side=tk.LEFT)

        self.se_canvas = tk.Canvas(bottom_frame, bg='yellow',
                                   cursor='bottom_right_corner',
                                   width=size, height=size)
        self.se_canvas.pack(side=tk.RIGHT)

        self.s_canvas = tk.Canvas(bottom_frame, bg='orange',
                                  cursor='bottom_side',
                                  height=size)
        self.s_canvas.pack(side=tk.TOP, fill=tk.X)


        # Left.
        self.w_canvas = tk.Canvas(self.window, bg='orange',
                                  cursor='left_side',
                                  width=size)
        self.w_canvas.pack(side=tk.LEFT, fill=tk.Y)


        # Right.
        self.e_canvas = tk.Canvas(self.window, bg='orange',
                                  cursor='right_side',
                                  width=size)
        self.e_canvas.pack(side=tk.RIGHT, fill=tk.Y)


        # Bind mouse events and set highlightthickness to 0.
        canvases = [self.drag_canvas, self.nw_canvas, self.n_canvas,
                    self.ne_canvas, self.e_canvas, self.se_canvas,
                    self.s_canvas, self.sw_canvas, self.w_canvas]
        for widget in canvases:
            widget.bind('<Button-1>', self.mouse_down)
            widget.bind('<B1-Motion>', self.mouse_drag)
            widget.config(highlightthickness=0)

        # Central frame.
        self.center_frame = tk.Frame(self.window, bg='plum1', borderwidth=0)
        self.center_frame.pack(side=tk.TOP, expand=True, fill=tk.BOTH)

        # Build the main non-border widgets.
        self.make_widgets()

        # Display the window.
        self.window.focus_force()
        self.window.mainloop()

    def kill_callback(self):
        # Destroy the tkinter window.
        if messagebox.askyesno('Close',
                               'Do you really want to close the program?'):
            self.window.destroy()

    def kill_mouse_enter(self, event):
        # Display the pirate cursor.
        self.kill_canvas.config(cursor='pirate')

    def kill_mouse_leave(self, event):
        # Display the normal cursor.
        self.kill_canvas.config(cursor=self.kill_canvas_cursor)

    def kill_clicked(self, event):
        '''The user clicked the close button. Kill the window.'''
        self.ignore_drag = True
        self.kill_callback()
        self.ignore_drag = False

    def mouse_down(self, event):
        '''The user pressed the mouse down on one of the handles.'''
        # Save the mouse position.
        self.start_x = event.x
        self.start_y = event.y

    def mouse_drag(self, event):
        '''The user is dragging over one of the handles.'''
        if self.ignore_drag:
            return

        # Get the window's current size and position.
        geometry = self.window.geometry().replace('+', 'x')
        wid, hgt, x, y = [int(value) for value in geometry.split('x')]

        # See how much the mouse has moved.
        dx = event.x - self.start_x
        dy = event.y - self.start_y

        # Update the window's size and position.
        widget = event.widget
        match widget:
            case self.drag_canvas:
                x += dx
                y += dy
            case self.nw_canvas:
                x += dx
                y += dy
                wid -= dx
                hgt -= dy
            case self.n_canvas:
                y += dy
                hgt -= dy
            case self.ne_canvas:
                y += dy
                wid += dx
                hgt -= dy
            case self.e_canvas:
                wid += dx
            case self.se_canvas:
                wid += dx
                hgt += dy
            case self.s_canvas:
                hgt += dy
            case self.w_canvas:
                x += dx
                wid -= dx
            case self.sw_canvas:
                x += dx
                wid -= dx
                hgt += dy

        # Update the window's size and position.
        self.window.geometry(f'{wid}x{hgt}+{x}+{y}')

        # Reposition the title text.
        if self.text_id is not None:
            tk.Tk.update(self.drag_canvas)
            x = self.drag_canvas.winfo_width() / 2
            y = self.drag_canvas.winfo_height() / 2
            self.drag_canvas.coords(self.text_id, x, y)

    def load_skin(self, path):
        '''Load the skin images in the given directory.'''
        # Make a new list to permanently hold images.
        self.photo_images = []

        # make sure the path ends with /.
        if not path.endswith('/'):
            path += '/'

        # Load the corner images.
        self.load_corner(self.ne_canvas, path + 'ne.png')
        self.load_corner(self.nw_canvas, path + 'nw.png')
        self.load_corner(self.sw_canvas, path + 'sw.png')
        self.load_corner(self.se_canvas, path + 'se.png')

        # Load the side images.
        self.load_side(self.w_canvas, path + 'w.png', True)
        self.load_side(self.e_canvas, path + 'e.png', True)
        self.load_side(self.n_canvas, path + 'n.png', False)
        self.load_side(self.s_canvas, path + 's.png', False)

        # Load the kill button.
        self.load_kill_image(self.nw_canvas, path + 'nw_close.png')
        self.load_kill_image(self.ne_canvas, path + 'ne_close.png')
        self.load_kill_image(self.sw_canvas, path + 'sw_close.png')
        self.load_kill_image(self.se_canvas, path + 'se_close.png')

        # Fill the drag canvas.
        self.load_filled(self.drag_canvas, path + 'drag.png')

        # Size the drag canvas.
        hgt = self.nw_canvas.winfo_height() - self.n_canvas.winfo_height()
        self.drag_canvas.config(height=hgt)

        # Set the center frame's color.
        image = Image.open(path + 'c.png')
        fill = rgb_to_tk_color(image.getpixel((0, 0)))
        self.center_frame.config(background=fill)

        # Draw text on drag_canvas.
        tk.Tk.update(self.drag_canvas)
        x = self.drag_canvas.winfo_width() / 2
        y = self.drag_canvas.winfo_height() / 2
        image = Image.open(path + 'title_color.png')
        fill = rgb_to_tk_color(image.getpixel((0, 0)))
        self.text_id = self.drag_canvas.create_text(
            x, y, text=self.window.title(), font=('Helvetica', 12, 'bold'),
            fill=fill, anchor=tk.CENTER)

    def load_corner(self, canvas, filepath):
        '''Load the image into the corner canvas and size the canvas to fit.'''
        # Delete any existing images.
        canvas.delete(tk.ALL)

        # Load and display the image.
        image = tk.PhotoImage(file=filepath) 
        self.photo_images.append(image)
        canvas.create_image(0, 0, image=image, anchor="nw")

        # Size the canvas to fit.
        canvas.config(width=image.width(), height=image.height())

    def load_side(self, canvas, filepath, is_vertical):
        '''Load the image into the side canvas and tile the image.'''
        # Delete any existing images.
        canvas.delete(tk.ALL)

        # Load the image.
        image = tk.PhotoImage(file=filepath) 
        self.photo_images.append(image)

        # Size the canvas to fit.
        if is_vertical:
            # Set the canvas's width.
            canvas.config(width=image.width())

            # Tile the image.
            tk.Tk.update(canvas)
            canvas_hgt = 2000
            image_hgt = image.height()
            for y in range(0, canvas_hgt, image_hgt):
                canvas.create_image(0, y, image=image, anchor="nw")
        else:
            # Set the canvas's height.
            canvas.config(height=image.height())

            # Tile the image.
            tk.Tk.update(canvas)
            canvas_wid = 2000
            image_wid = image.width()
            for x in range(0, canvas_wid, image_wid):
                canvas.create_image(x, 0, image=image, anchor="nw")

    def load_kill_image(self, canvas, filepath):
        '''Load and center the image in the canvas.'''
        # Do not delete existing images!

        # Try to load the image.
        try:
            image = tk.PhotoImage(file=filepath) 
            self.photo_images.append(image)
        except:
            return

        # Get canvas's dimensions.
        tk.Tk.update(canvas)
        canvas_wid = canvas.winfo_width()
        canvas_hgt = canvas.winfo_height()
        x = int((canvas_wid - image.width()) / 2)
        y = int((canvas_hgt - image.height()) / 2)

        # Display the image.
        kill_rect = canvas.create_image(x, y, image=image, anchor="nw")

        # Set the kill rectangle's mouse event handlers.
        canvas.tag_bind(kill_rect, '<Button-1>', self.kill_clicked)
        canvas.tag_bind(kill_rect, '<Enter>',    self.kill_mouse_enter)
        canvas.tag_bind(kill_rect, '<Leave>',    self.kill_mouse_leave)

        # Save a reference to this canvas and its normal cursor.
        self.kill_canvas = canvas
        self.kill_canvas_cursor = canvas.cget('cursor')

    def load_filled(self, canvas, filepath):
        '''Fill this canvas with the image.'''
        # Delete any existing images.
        canvas.delete(tk.ALL)
        tk.Tk.update(canvas)
        canvas_wid = 2000
        canvas_hgt = 2000

        # Load the image.
        image = tk.PhotoImage(file=filepath) 
        self.photo_images.append(image)
        image_wid = image.width()
        image_hgt = image.height()

        # Tile the canvas with the image.
        for y in range(0, canvas_hgt, image_hgt):
            for x in range(0, canvas_wid, image_wid):
                canvas.create_image(x, y, image=image, anchor="nw")

    def make_widgets(self):
        '''Make the main app widgets.'''
        self.skin_var = tk.StringVar(value='Safety')
        radio1 = tk.Radiobutton(self.center_frame, text='Safety',
                                variable=self.skin_var, value='Safety',
                                command=self.skin_selected)
        radio2 = tk.Radiobutton(self.center_frame, text='Pipes',
                                variable=self.skin_var, value='Pipes',
                                command=self.skin_selected)
        radio3 = tk.Radiobutton(self.center_frame, text='Rivets',
                                variable=self.skin_var, value='Rivets',
                                command=self.skin_selected)
        radio1.pack(side=tk.TOP, pady=(20,5))
        radio2.pack(side=tk.TOP, pady=5)
        radio3.pack(side=tk.TOP, pady=5)

        self.load_skin(self.skin_var.get())

    def skin_selected(self):
        '''Load the skin the user selected.'''
        self.load_skin(self.skin_var.get())

#%%
MoveBorderlessWindowApp()
print('Done')
