# -*- coding: utf-8 -*-
"""
Created on Thu Nov  7 15:31:13 2024

@author: mstep
"""
class Person:
    def __init__(self, index, preferences):
        names = [
            'Amy', 'Ben', 'Cam', 'Dan', 'Eve', 'Fae', 'Gar', 'Han', 'Ivy',
            'Jan', 'Kat', 'Lea', 'Mac', 'Nia', 'Oly', 'Pam', 'Qar', 'Rod',
            'Sid', 'Taj', 'Uli', 'Van', 'Win', 'Xia', 'Yaz', 'Zeb'
        ]
        self.name = names[index]
        self.preferences = preferences
        self.assignment = None

    def __str__(self):
        return f'{self.name}: {self.preferences}'

    def value(self):
        '''Return this person's value for their current assignment.'''
        return self.value_of(self.assignment)

    def value_of(self, assignment):
        '''Return the person's value for this assignment.'''
        # Return DONT_WANT if the person did not list this choice.
        DONT_WANT = 1000    # DONT_WANT is slightly better than NO_ASSIGNMENT
        NO_ASSIGNMENT = 1001
        if assignment is None:
            return NO_ASSIGNMENT
        if assignment not in self.preferences:
            return DONT_WANT
        return self.preferences.index(assignment)

# %%

import random
import tkinter as tk

class StableAppointmentsApp:
    def __init__(self):
        self.window = tk.Tk()
        self.window.title('stable_appointments')
        self.window.protocol('WM_DELETE_WINDOW', self.kill_callback)

        # Frame 1.
        frame_wid = 150
        frame_padx = 5
        frame_pady = 5
        label_pady = 5
        label_wid = 14
        frame = tk.Frame(self.window, borderwidth=5, relief=tk.GROOVE,
                         width=frame_wid, height=300)
        frame.pack_propagate(0)
        frame.pack(side=tk.LEFT, anchor=tk.N, padx=frame_padx, pady=frame_pady,
                   expand=False, fill=tk.Y)

        row_frame = tk.Frame(frame)
        row_frame.pack(side=tk.TOP, padx=5)
        label = tk.Label(row_frame, text='# People:', width=label_wid, anchor=tk.W)
        label.pack(side=tk.LEFT)

        self.num_people_entry = tk.Entry(row_frame, width=4, justify=tk.RIGHT)
        self.num_people_entry.pack(side=tk.LEFT)
        self.num_people_entry.insert(0, '5')

        row_frame = tk.Frame(frame)
        row_frame.pack(side=tk.TOP, padx=5)
        label = tk.Label(row_frame, text='# Preferences:', width=label_wid,
                         anchor=tk.W)
        label.pack(side=tk.LEFT)

        self.num_preferences_entry = tk.Entry(row_frame, width=4, justify=tk.RIGHT)
        self.num_preferences_entry.pack(side=tk.LEFT)
        self.num_preferences_entry.insert(0, '3')

        row_frame = tk.Frame(frame)
        row_frame.pack(side=tk.TOP, padx=5)
        label = tk.Label(row_frame, text='# Appointments:', width=label_wid,
                         anchor=tk.W)
        label.pack(side=tk.LEFT)

        self.num_appointments_entry = tk.Entry(row_frame, width=4, justify=tk.RIGHT)
        self.num_appointments_entry.pack(side=tk.LEFT)
        self.num_appointments_entry.insert(0, '5')

        button = tk.Button(frame, text='Randomize', width=10,
                           command=self.randomize)
        button.pack(side=tk.TOP, pady=10)

        button = tk.Button(frame, text='Assign', width=10,
                           command=self.assign)
        button.pack(side=tk.TOP, pady=10)

        self.score_label = tk.Label(frame)
        self.score_label.pack(side=tk.TOP)

        # Frame 2.
        frame = tk.Frame(self.window, borderwidth=5, relief=tk.GROOVE,
                         width=frame_wid)
        frame.pack_propagate(0) 
        frame.pack(side=tk.LEFT, anchor=tk.N, padx=frame_padx, pady=frame_pady,
                   expand=False, fill=tk.Y)

        label = tk.Label(frame, text='Preferences')
        label.pack(side=tk.TOP, pady=label_pady)

        self.preferences_list = tk.Listbox(frame)
        self.preferences_list.pack(side=tk.TOP, expand=True, fill=tk.Y,
                                   pady=(0,10))

        # Frame 3.
        frame = tk.Frame(self.window, borderwidth=5, relief=tk.GROOVE,
                         width=frame_wid)
        frame.pack_propagate(0) 
        frame.pack(side=tk.LEFT, anchor=tk.N, padx=frame_padx, pady=frame_pady,
                   expand=False, fill=tk.Y)

        label = tk.Label(frame, text='Appointments', pady=3)
        label.pack(side=tk.TOP, pady=label_pady)

        self.appointments_list = tk.Listbox(frame)
        self.appointments_list.pack(side=tk.TOP, expand=True, fill=tk.Y,
                                    pady=(0,10))

        # Frame 4.
        frame = tk.Frame(self.window, borderwidth=5, relief=tk.GROOVE,
                         width=frame_wid)
        frame.pack_propagate(0) 
        frame.pack(side=tk.LEFT, anchor=tk.N, padx=frame_padx, pady=frame_pady,
                   expand=False, fill=tk.Y)

        label = tk.Label(frame, text='People', pady=3)
        label.pack(side=tk.TOP, pady=label_pady)

        self.people_list = tk.Listbox(frame)
        self.people_list.pack(side=tk.TOP, expand=True, fill=tk.Y,
                                    pady=(0,10))

        # Bind the Return key.
        self.window.bind('<Return>', lambda e: self.randomize())
        self.num_people_entry.focus_force()
        self.window.mainloop()

    def kill_callback(self):
        # Destroy the tkinter window.
        self.window.destroy()

    def randomize(self):
        '''Generate random preferences.'''
        self.num_people = int(self.num_people_entry.get())
        self.num_preferences = int(self.num_preferences_entry.get())
        self.num_appointments = int(self.num_appointments_entry.get())
        self.people = []
        self.preferences_list.delete(0, tk.END)
        for i in range(self.num_people):
            preferences = random.sample(range(self.num_appointments),
                                        self.num_preferences)
            person = Person(i, preferences)
            self.people.append(person)
            self.preferences_list.insert(tk.END, person)
        self.appointments_list.delete(0, tk.END)
        self.people_list.delete(0, tk.END)

    def assign(self):
        '''Make assignments.'''
        num_assignments = int(self.num_appointments_entry.get())

        # Clear assignments.
        self.assigned_to = [None for i in range(num_assignments)]

        # Make initial assignments.
        for pref in range(self.num_preferences):
            # Try to assign this choice for the people.
            for person in self.people:
                # See if this Person has an assignment yet.
                if person.assignment is None:
                    # This Person is unassigned.
                    # See if this choice is available.
                    desired_choice = person.preferences[pref]
                    if self.assigned_to[desired_choice] is None:
                        # Assign this person to this choice.
                        self.assigned_to[desired_choice] = person
                        person.assignment = desired_choice

        # Assign anyone who doesn't have an assignment.
        for person in self.people:
            # See if this Person has an assignment.
            if person.assignment is None:
                # This Person is unassigned. Find an available choice.
                for i in range(len(self.assigned_to)):
                    if self.assigned_to[i] is None:
                        # Assign this Person.
                        self.assigned_to[i] = person
                        person.assignment = i
                        break

        # Try to improve the assignments.
        had_improvement = True
        while had_improvement:
            had_improvement = False

            # Look for profitable swaps.
            for person1 in self.people:
                for person2 in self.people:
                    assignment1 = person1.assignment
                    assignment2 = person2.assignment

                    # See if person1 and person2 should swap.
                    old_cost = person1.value() + person2.value()
                    new_cost = \
                        person1.value_of(assignment2) + \
                        person2.value_of(assignment1)
                    if new_cost < old_cost:
                        # Make the swap.
                        # print('Swapping:')
                        # print(f'    {person1} from {assignment1} to {assignment2}')
                        # print(f'    {person2} from {assignment2} to {assignment1}')
                        person1.assignment = assignment2
                        person2.assignment = assignment1
                        self.assigned_to[assignment1] = person2
                        self.assigned_to[assignment2] = person1
                        had_improvement = True

        # Display the assignments.
        self.show_assignments()

    def show_assignments(self):
        '''Display the assigments'''
        # Display the person assigned to each appointment.
        self.appointments_list.delete(0, tk.END)
        for i, person in enumerate(self.assigned_to):
            if person is None:
                self.appointments_list.insert(tk.END, '---')
            else:
                self.appointments_list.insert(tk.END, f'{i}: {person.name}')

        # Display each person's assigned appointment.
        self.people_list.delete(0, tk.END)
        score = 0
        for person in self.people:
            score += person.value()
            if person.assignment is None:
                self.people_list.insert(tk.END, f'{person.name}: ---')
            else:
                self.people_list.insert(tk.END,
                                        f'{person.name}: {person.assignment}')
        self.score_label['text'] = f'Score: {score}'

# In[3]:

StableAppointmentsApp()
print('Done')


# In[ ]:
